import subprocess
import sys

import torch

def get_subprocess_startup_info():
    if sys.platform != "win32":
        return None
    startup_info = subprocess.STARTUPINFO()
    startup_info.dwFlags |= subprocess.STARTF_USESHOWWINDOW
    return startup_info

def gpu_has_tensor_cores(device_index: int = 0) -> bool:
    """
    Check if the CUDA GPU has tensor cores for efficient FP16 operations.
    Returns False for CPUs, MPS devices, and older NVIDIA GPUs (< Volta architecture).
    GTX 16 series (Turing without tensor cores) is explicitly excluded.
    """
    if not torch.cuda.is_available():
        return False
    major, minor = torch.cuda.get_device_capability(device_index)
    # Tensor cores introduced in Volta (SM 7.0)
    if major < 7:
        return False
    if major > 7:
        return True
    # SM 7.x: Volta and Turing. GTX 16 series is Turing but lacks tensor cores
    name = torch.cuda.get_device_name(device_index).lower()
    if "gtx 16" in name:
        return False
    return True