"""
macOS 特定的工具函数
"""
import platform
import subprocess
import logging

logger = logging.getLogger(__name__)


def is_apple_silicon():
    """检查是否运行在 Apple Silicon 上"""
    try:
        return platform.machine() == 'arm64' and platform.system() == 'Darwin'
    except:
        return False


def get_macos_version():
    """获取 macOS 版本"""
    try:
        version = platform.mac_ver()[0]
        return tuple(map(int, version.split('.')))
    except:
        return None


def is_videotoolbox_available():
    """检查 VideoToolbox 是否可用"""
    try:
        result = subprocess.run(['ffmpeg', '-encoders'], 
                              capture_output=True, text=True, timeout=10)
        return 'videotoolbox' in result.stdout.lower()
    except:
        return False


def get_recommended_device():
    """获取推荐的设备"""
    import torch
    
    if torch.cuda.is_available():
        return "cuda:0"
    elif hasattr(torch.backends, 'mps') and torch.backends.mps.is_available():
        return "mps"
    else:
        return "cpu"


def get_recommended_codec():
    """获取推荐的编码器"""
    if is_videotoolbox_available():
        return "h264_videotoolbox"
    else:
        return "libx264"


def check_mps_compatibility():
    """检查 MPS 兼容性"""
    try:
        import torch
        if hasattr(torch.backends, 'mps') and torch.backends.mps.is_available():
            # 测试基本的 MPS 操作
            test_tensor = torch.randn(2, 2, device='mps')
            test_result = test_tensor + 1
            
            # 测试 grid_sample（这是经常出问题的地方）
            import torch.nn.functional as F
            input_tensor = torch.randn(1, 3, 4, 4, device='mps')
            grid = torch.randn(1, 2, 2, 2, device='mps')
            try:
                result = F.grid_sample(input_tensor, grid, align_corners=False)
                return True
            except Exception as grid_error:
                logger.warning(f"MPS grid_sample test failed: {grid_error}")
                return False
    except Exception as e:
        logger.warning(f"MPS compatibility test failed: {e}")
        return False
    return False


def should_use_mps():
    """判断是否应该使用 MPS"""
    import os
    
    # 检查环境变量强制设置
    force_cpu = os.environ.get('LADA_FORCE_CPU', '0').lower() in ('1', 'true', 'yes')
    if force_cpu:
        logger.info("Forced to use CPU due to LADA_FORCE_CPU environment variable")
        return False
    
    # 检查 MPS 兼容性
    if not check_mps_compatibility():
        logger.warning("MPS compatibility issues detected, recommending CPU fallback")
        return False
    
    return True


def optimize_for_apple_silicon():
    """为 Apple Silicon 优化设置"""
    import os
    
    if is_apple_silicon():
        # 启用 MPS 回退
        os.environ.setdefault('PYTORCH_ENABLE_MPS_FALLBACK', '1')
        
        # 优化内存使用
        os.environ.setdefault('PYTORCH_MPS_HIGH_WATERMARK_RATIO', '0.0')
        
        # 检查 MPS 兼容性
        if not check_mps_compatibility():
            logger.warning("MPS compatibility issues detected, consider using CPU fallback")
        
        logger.info("Applied Apple Silicon optimizations")
    
    # macOS GUI 优化
    if platform.system() == 'Darwin':
        # 设置 GTK 主题
        os.environ.setdefault('GTK_THEME', 'Adwaita:dark')
        
        # 启用 macOS 集成
        os.environ.setdefault('GTK_CSD', '1')
        
        # 设置 DPI 缩放
        os.environ.setdefault('GDK_SCALE', '1')
        
        # 强制使用 macOS 后端
        os.environ.setdefault('GDK_BACKEND', 'macos')
        
        # 禁用 GTK3 以避免冲突
        os.environ.setdefault('GTK_MODULES', '')
        
        # 禁用硬件加速以避免 OpenGL 问题
        os.environ.setdefault('GDK_RENDERING', 'image')
        os.environ.setdefault('GSK_RENDERER', 'cairo')
        
        # 禁用 OpenGL 调试
        os.environ.setdefault('G_ENABLE_DIAGNOSTIC', '0')
        
        logger.info("Applied macOS GUI optimizations")


def check_gui_environment():
    """检查GUI环境"""
    import os
    
    gui_info = {
        'display_available': False,
        'gtk_available': False,
        'adwaita_available': False,
        'display_env': None
    }
    
    # 检查显示环境
    if 'DISPLAY' in os.environ:
        gui_info['display_env'] = 'X11'
        gui_info['display_available'] = True
    elif 'WAYLAND_DISPLAY' in os.environ:
        gui_info['display_env'] = 'Wayland'
        gui_info['display_available'] = True
    elif platform.system() == 'Darwin':
        gui_info['display_env'] = 'Quartz'
        gui_info['display_available'] = True
    
    # 检查 GTK
    try:
        import gi
        gi.require_version('Gtk', '4.0')
        from gi.repository import Gtk
        gui_info['gtk_available'] = True
    except (ImportError, ValueError):
        pass
    
    # 检查 Adwaita
    try:
        import gi
        gi.require_version('Adw', '1')
        from gi.repository import Adw
        gui_info['adwaita_available'] = True
    except (ImportError, ValueError):
        pass
    
    return gui_info


def get_system_info():
    """获取系统信息"""
    info = {
        'platform': platform.system(),
        'machine': platform.machine(),
        'is_apple_silicon': is_apple_silicon(),
        'macos_version': get_macos_version(),
        'videotoolbox_available': is_videotoolbox_available(),
        'gui_info': check_gui_environment(),
    }
    
    try:
        import torch
        info['mps_available'] = hasattr(torch.backends, 'mps') and torch.backends.mps.is_available()
        info['cuda_available'] = torch.cuda.is_available()
    except ImportError:
        info['mps_available'] = False
        info['cuda_available'] = False
    
    return info