import logging

import av
import io
import os
import subprocess
import shutil
from typing import Optional
from lada.lib import video_utils, os_utils

logger = logging.getLogger(__name__)

def combine_audio_video_files(av_video_metadata: video_utils.VideoMetadata, tmp_v_video_input_path, av_video_output_path):
    audio_codec = get_audio_codec(av_video_metadata.video_file)
    if audio_codec:
        # Only copy AAC, re-encode everything else (MP3, WMA, etc.) to AAC 192 kbps
        is_aac = audio_codec == "aac"
        needs_video_delay = av_video_metadata.start_pts > 0

        # Detect video codec from temp file to set correct tag
        video_codec = get_video_codec(tmp_v_video_input_path)
        video_tag = get_quicktime_video_tag(video_codec)

        cmd = ["ffmpeg", "-y", "-loglevel", "quiet"]
        cmd += ["-i", av_video_metadata.video_file]
        if needs_video_delay > 0:
            delay_in_seconds = float(av_video_metadata.start_pts * av_video_metadata.time_base)
            cmd += ["-itsoffset", str(delay_in_seconds)]
        cmd += ["-i", tmp_v_video_input_path]
        if is_aac:
            # Copy AAC audio without re-encoding
            cmd += ["-c", "copy"]
        else:
            # Re-encode non-AAC audio to AAC (MP3, WMA, obscure codecs, etc.)
            cmd += ["-c:v", "copy"]
            cmd += ["-c:a", "aac", "-b:a", "192k"]
        cmd += ["-map", "1:v:0"]
        cmd += ["-map", "0:a:0"]
        if video_tag:
            cmd += ["-tag:v", video_tag]  # QuickTime compatible tag (avc1 for H.264, hvc1 for HEVC)
        cmd += ["-movflags", "+faststart"]
        cmd += [av_video_output_path]
        result = subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                              startupinfo=os_utils.get_subprocess_startup_info())

        if result.returncode != 0:
            error_msg = result.stderr.decode('utf-8', errors='ignore')
            logger.error(f"FFmpeg failed to combine audio/video: {error_msg}")
            raise RuntimeError(f"FFmpeg audio combine failed: {error_msg}")
    else:
        # No audio - remux with faststart for QuickTime compatibility
        video_codec = get_video_codec(tmp_v_video_input_path)
        video_tag = get_quicktime_video_tag(video_codec)

        cmd = ["ffmpeg", "-y", "-loglevel", "quiet"]
        cmd += ["-i", tmp_v_video_input_path]
        cmd += ["-c", "copy"]
        if video_tag:
            cmd += ["-tag:v", video_tag]  # QuickTime compatible tag (avc1 for H.264, hvc1 for HEVC)
        cmd += ["-movflags", "+faststart"]
        cmd += [av_video_output_path]
        result = subprocess.run(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE,
                              startupinfo=os_utils.get_subprocess_startup_info())

        if result.returncode != 0:
            error_msg = result.stderr.decode('utf-8', errors='ignore')
            logger.error(f"FFmpeg failed to remux video: {error_msg}")
            raise RuntimeError(f"FFmpeg remux failed: {error_msg}")

    # Only remove temp file if output was successfully created with valid size
    if os.path.exists(av_video_output_path) and os.path.getsize(av_video_output_path) > 0:
        os.remove(tmp_v_video_input_path)
        logger.info(f"Successfully created output: {av_video_output_path}")
    else:
        raise RuntimeError(f"Output file not created or is empty. Temp file preserved at: {tmp_v_video_input_path}")

def get_audio_codec(file_path: str) -> Optional[str]:
    cmd = f"ffprobe -loglevel error -select_streams a:0 -show_entries stream=codec_name -of default=nw=1:nk=1"
    cmd = cmd.split() + [file_path]
    cmd_result = subprocess.run(cmd, stdout=subprocess.PIPE, startupinfo=os_utils.get_subprocess_startup_info())
    audio_codec = cmd_result.stdout.decode('utf-8').strip().lower()
    return audio_codec if len(audio_codec) > 0 else None

def get_video_codec(file_path: str) -> Optional[str]:
    """Detect video codec from file using ffprobe."""
    cmd = f"ffprobe -loglevel error -select_streams v:0 -show_entries stream=codec_name -of default=nw=1:nk=1"
    cmd = cmd.split() + [file_path]
    cmd_result = subprocess.run(cmd, stdout=subprocess.PIPE, startupinfo=os_utils.get_subprocess_startup_info())
    video_codec = cmd_result.stdout.decode('utf-8').strip().lower()
    return video_codec if len(video_codec) > 0 else None

def get_quicktime_video_tag(video_codec: Optional[str]) -> Optional[str]:
    """Get the appropriate QuickTime video tag for the codec."""
    if not video_codec:
        return None

    # H.264 variants
    if video_codec in ('h264', 'avc', 'h264_videotoolbox'):
        return "avc1"
    # HEVC/H.265 variants
    elif video_codec in ('hevc', 'h265', 'hvc1', 'hevc_videotoolbox'):
        return "hvc1"

    return None

def is_output_container_compatible_with_input_audio_codec(audio_codec: str, output_path: str) -> bool:
    file_extension = os.path.splitext(output_path)[1]
    file_extension = file_extension.lower()
    if file_extension in ('.mp4', '.m4v'):
        output_container_format = "mp4"
    elif file_extension == '.mkv':
        output_container_format = "matroska"
    else:
        logger.info(f"Couldn't determine video container format based on file extension: {file_extension}")
        return False

    buf = io.BytesIO()
    with av.open(buf, 'w', output_container_format) as container:
        return audio_codec in container.supported_codecs
