import logging
import os
import platform
import subprocess
from gettext import gettext as _

import torch
from gi.repository import Gio
from gi.repository import Gtk, GLib, Gdk

from lada import LOG_LEVEL
from lada.lib import video_utils

logger = logging.getLogger(__name__)
logging.basicConfig(level=LOG_LEVEL)

# Minimum RAM (in GB) required for MPS acceleration
# Systems with less RAM experience crashes due to MPS threading issues
# v0.8.6: Lowered from 24GB to 8GB - 16GB M1 confirmed working after timestamp fix
MPS_MIN_RAM_GB = 8


def get_system_ram_gb() -> float:
    """Get total system RAM in GB. Returns 0 if unable to detect."""
    try:
        if platform.system() == 'Darwin':
            result = subprocess.run(['sysctl', '-n', 'hw.memsize'],
                                  capture_output=True, text=True, timeout=5)
            total_mem_bytes = int(result.stdout.strip())
            return total_mem_bytes / (1024**3)
        elif platform.system() == 'Linux':
            with open('/proc/meminfo', 'r') as f:
                for line in f:
                    if line.startswith('MemTotal:'):
                        # MemTotal is in kB
                        kb = int(line.split()[1])
                        return kb / (1024**2)
        elif platform.system() == 'Windows':
            import ctypes
            kernel32 = ctypes.windll.kernel32
            c_ulong = ctypes.c_ulong
            class MEMORYSTATUS(ctypes.Structure):
                _fields_ = [
                    ('dwLength', c_ulong),
                    ('dwMemoryLoad', c_ulong),
                    ('dwTotalPhys', c_ulong),
                    ('dwAvailPhys', c_ulong),
                    ('dwTotalPageFile', c_ulong),
                    ('dwAvailPageFile', c_ulong),
                    ('dwTotalVirtual', c_ulong),
                    ('dwAvailVirtual', c_ulong)
                ]
            memStatus = MEMORYSTATUS()
            memStatus.dwLength = ctypes.sizeof(MEMORYSTATUS)
            kernel32.GlobalMemoryStatus(ctypes.byref(memStatus))
            return memStatus.dwTotalPhys / (1024**3)
    except Exception as e:
        logger.warning(f"Could not detect system RAM: {e}")
    return 0


def is_mps_supported() -> bool:
    """Check if MPS is available AND the system has enough RAM to use it safely."""
    if not (hasattr(torch.backends, 'mps') and torch.backends.mps.is_available()):
        return False

    ram_gb = get_system_ram_gb()
    if ram_gb > 0 and ram_gb < MPS_MIN_RAM_GB:
        logger.warning(f"MPS disabled: System has {ram_gb:.1f}GB RAM, minimum {MPS_MIN_RAM_GB}GB required for stability")
        return False

    return True

def is_device_available(device: str) -> bool:
    device = device.lower()
    if device == 'cpu':
        return True
    elif device.startswith("cuda:"):
        return torch.cuda.is_available() and device_to_gpu_id(device) < torch.cuda.device_count()
    elif device == 'mps':
        return is_mps_supported()
    return False


def device_to_gpu_id(device) -> int | None:
    if device.startswith("cuda:"):
        return int(device.split(":")[-1])
    return None


def get_available_gpus():
    gpus = []
    
    # CUDA GPUs
    if torch.cuda.is_available():
        for id in range(torch.cuda.device_count()):
            gpu_name = torch.cuda.get_device_properties(id).name
            # We're using these GPU names in a ComboBox but libadwaita sets up the label with max-width-chars: 20 and there does not
            # seem to be a way to overwrite this. So let's try to make sure GPU names are below 20 characters to be readable
            if gpu_name.startswith("NVIDIA GeForce RTX"):
                gpu_name = gpu_name.replace("NVIDIA GeForce RTX", "RTX")
            gpus.append((f"cuda:{id}", gpu_name))
    
    # Apple Silicon MPS (only if system has enough RAM)
    if is_mps_supported():
        gpus.append(("mps", "Apple Silicon GPU"))
    
    return gpus

def skip_if_uninitialized(f):
    def noop(*args):
        return
    def wrapper(*args):
        return f(*args) if args[0].init_done else noop
    return wrapper

def get_available_video_codecs() -> list[str]:
    filter_list = ['libx264', 'h264_nvenc', 'libx265', 'hevc_nvenc', 'libsvtav1', 'librav1e', 'libaom-av1', 'av1_nvenc', 'h264_videotoolbox', 'hevc_videotoolbox']
    return [codec_short_name for codec_short_name, codec_long_name in video_utils.get_available_video_encoder_codecs() if codec_short_name in filter_list]

def validate_file_name_pattern(file_name_pattern: str) -> bool:
    if not "{orig_file_name}" in file_name_pattern:
        return False
    if os.sep in file_name_pattern:
        return False
    file_extension = os.path.splitext(file_name_pattern)[1].lower()
    if file_extension not in [".mp4", ".mkv", ".mov", ".m4v"]:
        return False
    return True

def filter_video_files(files: list[Gio.File]) -> list[Gio.File]:
    def is_video_file(file: Gio.File):
        file_info: Gio.FileInfo = file.query_info("standard::content-type", Gio.FileQueryInfoFlags.NONE)
        content_type = file_info.get_content_type() # on linux content_type is MIME type but on windows it's just a file extension
        if content_type is None: return False
        mime_type = Gio.content_type_get_mime_type(content_type)
        if mime_type is None: return False
        return mime_type.startswith("video/")
    filtered_files = [file for file in files if is_video_file(file)]
    return filtered_files

def show_open_files_dialog(callback, dismissed_callback):
    file_dialog = Gtk.FileDialog()
    video_file_filter = Gtk.FileFilter()
    video_file_filter.add_mime_type("video/*")
    file_dialog.set_default_filter(video_file_filter)
    file_dialog.set_title(_("Select one or multiple video files"))
    def on_open_multiple(_file_dialog, result):
        try:
            video_files = _file_dialog.open_multiple_finish(result)
            if len(video_files) > 0:
                callback(video_files)
        except GLib.Error as error:
            if error.message == "Dismissed by user":
                dismissed_callback()
                logger.debug("FileDialog cancelled: Dismissed by user")
            else:
                logger.error(f"Error opening file: {error.message}")
                raise error
    file_dialog.open_multiple(callback=on_open_multiple)

def create_video_files_drop_target(callback):
    drop_target = Gtk.DropTarget.new(Gio.File, actions=Gdk.DragAction.COPY)
    drop_target.set_gtypes((Gdk.FileList,))
    def on_file_drop(_drop_target, files: list[Gio.File], x, y):
        video_files = filter_video_files(files)
        if len(video_files) > 0:
            callback(video_files)
    drop_target.connect("drop", on_file_drop)
    return drop_target
