import gettext
import locale
import logging
import pathlib
import sys

import gi

from lada import VERSION, LOG_LEVEL, LOCALE_DIR

gi.require_version('Gtk', '4.0')
gi.require_version('Adw', '1')
gi.require_version('Gst', '1.0')
gi.require_version('GstApp', '1.0')

from gi.repository import Gtk, Gio, Adw, Gdk, Gst, GObject, GLib

here = pathlib.Path(__file__).parent.resolve()

logger = logging.getLogger(__name__)
logging.basicConfig(level=LOG_LEVEL)

from lada.gui.window import MainWindow
from lada.gui.shortcuts import ShortcutsWindow, ShortcutsManager
from lada.gui.config.config import Config
from lada.lib.macos_utils import optimize_for_apple_silicon

class LadaApplication(Adw.Application):

    def __init__(self):
        # 首先检查显示是否可用
        display = Gdk.Display.get_default()
        if display is None:
            logger.error("No display available. GTK cannot initialize.")
            raise RuntimeError("No display available. Please check your GTK installation and display environment.")
        
        # 使用 HANDLES_OPEN 标志来处理文件打开
        super().__init__(application_id='io.github.ladaapp.lada',
                         flags=Gio.ApplicationFlags.HANDLES_OPEN)
        
        logger.info(f"Display available: {display}")
        
        self.create_action('quit', self.on_close, ['<primary>q'])
        self.create_action('about', self.on_about_action)
        self.create_action('shortcuts', self.on_shortcut_action)

        # 安全地设置 CSS 提供者
        try:
            css_provider = Gtk.CssProvider()
            css_provider.load_from_path(str(here.joinpath('style.css')))
            Gtk.StyleContext.add_provider_for_display(display, css_provider, Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION)
            logger.info("CSS provider loaded successfully")
        except Exception as e:
            logger.error(f"Failed to setup CSS provider: {e}")

        try:
            resource = Gio.resource_load(str(here.joinpath('resources.gresource')))
            Gio.resources_register(resource)
            logger.info("Resources loaded successfully")
        except Exception as e:
            logger.error(f"Failed to load resources: {e}")

        self._shortcuts_manager: ShortcutsManager = ShortcutsManager()
        self._config: Config = Config(self.get_style_manager())
        self._config.load_config()
        self.window: MainWindow | None = None

        try:
            Gst.init(None)
            logger.info("GStreamer initialized successfully")
        except Exception as e:
            logger.warning(f"GStreamer initialization failed: {e}")

    @GObject.Property(type=ShortcutsManager)
    def shortcuts_manager(self):
        return self._shortcuts_manager

    @shortcuts_manager.setter
    def shortcuts_manager(self, value):
        self._shortcuts_manager = value

    @GObject.Property(type=Config)
    def config(self):
        return self._config

    @config.setter
    def config(self, value):
        self._config = value

    def do_activate(self):
        logger.info("Activating application...")
        win = self.props.active_window
        if not win:
            logger.info("Creating new main window...")
            try:
                win = MainWindow(application=self)
                logger.info("Main window created successfully")
                
                # 绑定属性
                self.bind_property("style-manager", win.preview_view.widget_timeline, "style-manager", flags=GObject.BindingFlags.SYNC_CREATE)
                self.bind_property("shortcuts-manager", win.preview_view, "shortcuts-manager", flags=GObject.BindingFlags.SYNC_CREATE)
                self.bind_property("config", win.preview_view, "config", flags=GObject.BindingFlags.SYNC_CREATE)
                self.bind_property("config", win.export_view, "config", flags=GObject.BindingFlags.SYNC_CREATE)
                self.bind_property("config", win, "config", flags=GObject.BindingFlags.SYNC_CREATE)
                self.bind_property("shortcuts-manager", win.preview_view, "shortcuts-manager", flags=GObject.BindingFlags.SYNC_CREATE)
                self.bind_property("shortcuts-manager", win, "shortcuts-manager", flags=GObject.BindingFlags.SYNC_CREATE)
                self.window = win

                self._shortcuts_manager.init(win.shortcut_controller)
                logger.info("Window setup completed")
                
            except Exception as e:
                logger.error(f"Failed to create main window: {e}")
                import traceback
                traceback.print_exc()
                return
        
        logger.info("Presenting window...")
        win.present()
        
        # macOS 特定：确保窗口获得焦点
        import platform
        if platform.system() == 'Darwin':
            win.set_focus_visible(True)
            GLib.timeout_add(100, lambda: win.present() or False)  # 延迟再次present

    def do_open(self, files, n_files, hint):
        """处理文件打开"""
        logger.info(f"Opening {n_files} files")
        self.do_activate()
        
        if self.window and n_files > 0:
            # 将文件传递给主窗口处理
            try:
                video_files = [f for f in files if f.get_path()]
                if video_files:
                    # 这里可以添加文件处理逻辑
                    logger.info(f"Processing {len(video_files)} video files")
            except Exception as e:
                logger.error(f"Error processing files: {e}")

    def on_close(self, *args):
        if self.window:
            self.window.close()
            self.quit()

    def on_about_action(self, *args):
        about = Adw.AboutDialog(application_name='lada',
                                application_icon='io.github.ladaapp.lada',
                                license_type=Gtk.License.AGPL_3_0,
                                website='https://github.com/ladaapp',
                                issue_url='https://github.com/ladaapp/issues',
                                version=VERSION)
        about.present(self.props.active_window)

    def on_shortcut_action(self, *args):
        shortcuts_window = ShortcutsWindow(self._shortcuts_manager)
        shortcuts_window.show()

    def create_action(self, name, callback, shortcuts=None):
        action = Gio.SimpleAction.new(name, None)
        action.connect("activate", callback)
        self.add_action(action)
        if shortcuts:
            self.set_accels_for_action(f"app.{name}", shortcuts)

def init_localization():
    APP_NAME = 'lada'
    try:
        locale.bindtextdomain(APP_NAME, LOCALE_DIR)
        locale.textdomain(APP_NAME)
    except AttributeError as e:
        pass
        # TODO: Workaround for Windows as reported in #88
        #  Translations of .ui files will probably not work then
    gettext.bindtextdomain(APP_NAME, LOCALE_DIR)
    gettext.textdomain(APP_NAME)

def main():
    logger.info("Starting Lada GUI...")
    
    # 检查显示环境
    import os
    if 'DISPLAY' not in os.environ and 'WAYLAND_DISPLAY' not in os.environ:
        logger.warning("No display environment detected")
    
    init_localization()
    optimize_for_apple_silicon()  # 应用 Apple Silicon 优化
    
    # macOS 特定设置
    import platform
    if platform.system() == 'Darwin':
        logger.info("Detected macOS, applying macOS-specific settings...")
        # 设置 macOS 应用程序名称
        try:
            from Foundation import NSBundle
            bundle = NSBundle.mainBundle()
            if bundle:
                info = bundle.localizedInfoDictionary() or bundle.infoDictionary()
                if info:
                    info['CFBundleName'] = 'Lada'
        except ImportError:
            logger.debug("Foundation not available, skipping bundle name setup")
    
    app = LadaApplication()
    logger.info("Application created, starting main loop...")
    
    try:
        exit_code = app.run(sys.argv)
        logger.info(f"Application exited with code: {exit_code}")
        return exit_code
    except KeyboardInterrupt:
        logger.info("Received Ctrl-C, quitting")
        app.on_close()
        return 0
    except Exception as e:
        logger.error(f"Application error: {e}")
        import traceback
        traceback.print_exc()
        return 1

if __name__ == "__main__":
    main()
